<?php
require_once '../config/config.php';

if (!isAdmin()) {
    redirect('../dashboard.php');
}

$db = Database::getInstance()->getConnection();

/**
 * Fetch rainfall data from BMKG API
 */
function fetchBMKGData($api_url) {
    if (empty($api_url)) {
        return null;
    }
    
    // Initialize cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'SumberAI/1.0');
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200 || empty($response)) {
        return null;
    }
    
    $data = json_decode($response, true);
    return $data;
}

/**
 * Extract rainfall value from BMKG API response
 * BMKG API format may vary, so we try multiple common formats
 */
function extractRainfallFromBMKG($bmkg_data) {
    if (empty($bmkg_data)) {
        return null;
    }
    
    // Try different possible BMKG API response formats
    $rainfall = null;
    
    // Format 1: Direct values
    if (isset($bmkg_data['rainfall'])) {
        $rainfall = floatval($bmkg_data['rainfall']);
    } elseif (isset($bmkg_data['precipitation'])) {
        $rainfall = floatval($bmkg_data['precipitation']);
    } elseif (isset($bmkg_data['curah_hujan'])) {
        $rainfall = floatval($bmkg_data['curah_hujan']);
    } elseif (isset($bmkg_data['rr'])) { // BMKG sometimes uses 'rr' for rainfall
        $rainfall = floatval($bmkg_data['rr']);
    }
    
    // Format 2: Nested in 'data' object
    if ($rainfall === null && isset($bmkg_data['data'])) {
        if (isset($bmkg_data['data']['rainfall'])) {
            $rainfall = floatval($bmkg_data['data']['rainfall']);
        } elseif (isset($bmkg_data['data']['precipitation'])) {
            $rainfall = floatval($bmkg_data['data']['precipitation']);
        } elseif (isset($bmkg_data['data']['curah_hujan'])) {
            $rainfall = floatval($bmkg_data['data']['curah_hujan']);
        } elseif (isset($bmkg_data['data']['rr'])) {
            $rainfall = floatval($bmkg_data['data']['rr']);
        }
    }
    
    // Format 3: Array format (first element)
    if ($rainfall === null && isset($bmkg_data['data']) && is_array($bmkg_data['data']) && !empty($bmkg_data['data'][0])) {
        $first_item = $bmkg_data['data'][0];
        if (isset($first_item['rainfall'])) {
            $rainfall = floatval($first_item['rainfall']);
        } elseif (isset($first_item['precipitation'])) {
            $rainfall = floatval($first_item['precipitation']);
        } elseif (isset($first_item['curah_hujan'])) {
            $rainfall = floatval($first_item['curah_hujan']);
        } elseif (isset($first_item['rr'])) {
            $rainfall = floatval($first_item['rr']);
        }
    }
    
    // Format 4: Try to find any numeric value that might be rainfall
    // (Last resort - look for values in common keys)
    if ($rainfall === null) {
        $possible_keys = ['rainfall', 'precipitation', 'curah_hujan', 'rr', 'rain', 'hujan'];
        foreach ($possible_keys as $key) {
            if (isset($bmkg_data[$key]) && is_numeric($bmkg_data[$key])) {
                $rainfall = floatval($bmkg_data[$key]);
                break;
            }
        }
    }
    
    return $rainfall;
}

// Get all dusun that are properly linked to desa (only data input by admin)
// This ensures we only predict for valid desa/dusun that admin has created
// CRITICAL: Only predict for dusun that are linked to desa created by admin
// This synchronizes predictions with admin input data
$stmt = $db->query("SELECT d.*, des.nama_desa, des.id as desa_id_real, des.kode_desa
                    FROM dusun d 
                    INNER JOIN desa des ON d.desa_id = des.id 
                    WHERE d.desa_id IS NOT NULL AND d.desa_id > 0
                    ORDER BY des.nama_desa, d.nama_dusun");
$dusun_list = $stmt->fetchAll();

// Log for debugging (optional - can be removed in production)
if (empty($dusun_list)) {
    $_SESSION['warning'] = 'Tidak ada dusun yang terhubung dengan desa. Pastikan admin sudah menginput data desa dan dusun.';
}

$generated = 0;
$errors = 0;
$bmkg_updated = 0;

foreach ($dusun_list as $dusun) {
    // Try to fetch data from BMKG API if URL is available
    $bmkg_rainfall = null;
    if (!empty($dusun['api_bmkg_url'])) {
        $bmkg_data = fetchBMKGData($dusun['api_bmkg_url']);
        if ($bmkg_data) {
            $bmkg_rainfall = extractRainfallFromBMKG($bmkg_data);
            
            // Save BMKG data to database if we got valid data
            if ($bmkg_rainfall !== null && $bmkg_rainfall >= 0) {
                $today = date('Y-m-d');
                $stmt_check = $db->prepare("SELECT id FROM data_curah_hujan WHERE dusun_id = ? AND tanggal = ? AND sumber_data = 'bmkg'");
                $stmt_check->execute([$dusun['id'], $today]);
                
                if ($stmt_check->fetch()) {
                    // Update existing
                    $stmt_update = $db->prepare("UPDATE data_curah_hujan SET curah_hujan = ? WHERE dusun_id = ? AND tanggal = ? AND sumber_data = 'bmkg'");
                    $stmt_update->execute([$bmkg_rainfall, $dusun['id'], $today]);
                } else {
                    // Insert new
                    $stmt_insert = $db->prepare("INSERT INTO data_curah_hujan (dusun_id, tanggal, curah_hujan, sumber_data) VALUES (?, ?, ?, 'bmkg')");
                    $stmt_insert->execute([$dusun['id'], $today, $bmkg_rainfall]);
                }
                $bmkg_updated++;
            }
        }
    }
    
    // Get recent rainfall data (last 30 days)
    // Priority: manual (admin input) > BMKG > sensor
    // Manual data is most trusted because admin directly inputs it
    
    // First, try to get manual data (input by admin)
    $stmt = $db->prepare("SELECT AVG(curah_hujan) as avg_rain, COUNT(*) as count 
                          FROM data_curah_hujan 
                          WHERE dusun_id = ? AND tanggal >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
                          AND sumber_data = 'manual'");
    $stmt->execute([$dusun['id']]);
    $rain_data_manual = $stmt->fetch();
    
    // If no manual data, try BMKG
    if (empty($rain_data_manual['count']) || $rain_data_manual['count'] == 0) {
        $stmt = $db->prepare("SELECT AVG(curah_hujan) as avg_rain, COUNT(*) as count 
                              FROM data_curah_hujan 
                              WHERE dusun_id = ? AND tanggal >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
                              AND sumber_data = 'bmkg'");
        $stmt->execute([$dusun['id']]);
        $rain_data_bmkg = $stmt->fetch();
        
        // If no BMKG data, try sensor or all data
        if (empty($rain_data_bmkg['count']) || $rain_data_bmkg['count'] == 0) {
            $stmt = $db->prepare("SELECT AVG(curah_hujan) as avg_rain, COUNT(*) as count 
                                  FROM data_curah_hujan 
                                  WHERE dusun_id = ? AND tanggal >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)");
            $stmt->execute([$dusun['id']]);
            $rain_data = $stmt->fetch();
        } else {
            $rain_data = $rain_data_bmkg;
        }
    } else {
        // Use manual data (highest priority)
        $rain_data = $rain_data_manual;
    }
    
    // Get recent reports - only verified reports count (data from admin verification)
    // Negative reports (kekeringan) increase risk
    $stmt = $db->prepare("SELECT COUNT(*) FROM laporan_warga 
                          WHERE dusun_id = ? AND jenis_laporan = 'kekeringan' 
                          AND status = 'verified'
                          AND tanggal >= DATE_SUB(CURDATE(), INTERVAL 14 DAY)");
    $stmt->execute([$dusun['id']]);
    $negative_report_count = $stmt->fetchColumn();
    
    // Positive reports (ketersediaan_air, kualitas_air baik) decrease risk
    // These reports indicate water is available, so reduce risk significantly
    $stmt = $db->prepare("SELECT COUNT(*) FROM laporan_warga 
                          WHERE dusun_id = ? AND jenis_laporan IN ('ketersediaan_air', 'kualitas_air', 'penampungan', 'sumur', 'pam')
                          AND status = 'verified'
                          AND tanggal >= DATE_SUB(CURDATE(), INTERVAL 14 DAY)");
    $stmt->execute([$dusun['id']]);
    $positive_report_count = $stmt->fetchColumn();
    
    // Get total verified reports for this dusun (to understand data completeness)
    $stmt = $db->prepare("SELECT COUNT(*) FROM laporan_warga 
                          WHERE dusun_id = ? AND status = 'verified'
                          AND tanggal >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)");
    $stmt->execute([$dusun['id']]);
    $total_verified_reports = $stmt->fetchColumn();
    
    // Get total verified warga in this dusun (from registration data)
    $stmt = $db->prepare("SELECT COUNT(*) FROM users 
                          WHERE dusun_id = ? AND role = 'warga' AND status = 'active'");
    $stmt->execute([$dusun['id']]);
    $total_warga = $stmt->fetchColumn();
    
    // Calculate report factor: negative reports increase risk, positive reports decrease risk
    $report_count = $negative_report_count;
    
    // Calculate risk score (improved AI algorithm)
    $avg_rain = floatval($rain_data['avg_rain'] ?? 0);
    $data_count = intval($rain_data['count'] ?? 0);
    
    // Priority: Manual data (admin input) > BMKG > Historical average
    // Check if we have manual data (admin input) - this is most trusted
    $stmt_manual_today = $db->prepare("SELECT curah_hujan FROM data_curah_hujan 
                                       WHERE dusun_id = ? AND tanggal = CURDATE() 
                                       AND sumber_data = 'manual' 
                                       ORDER BY created_at DESC LIMIT 1");
    $stmt_manual_today->execute([$dusun['id']]);
    $manual_rainfall_today = $stmt_manual_today->fetchColumn();
    
    if ($manual_rainfall_today !== false && $manual_rainfall_today !== null) {
        // Admin has manually input data today - this is most trusted
        // Admin input takes highest priority over everything else
        if ($data_count > 0 && $avg_rain > 0) {
            // Weight: 85% manual (admin input today), 15% historical average
            $avg_rain = ($manual_rainfall_today * 0.85) + ($avg_rain * 0.15);
        } else {
            // No historical data, use manual directly (admin knows best)
            $avg_rain = floatval($manual_rainfall_today);
        }
    } elseif ($bmkg_rainfall !== null && $bmkg_rainfall >= 0) {
        // Use BMKG data if available (but lower priority than manual)
        if ($data_count > 0 && $avg_rain > 0) {
            // Weight: 70% BMKG (today's data), 30% historical average
            $avg_rain = ($bmkg_rainfall * 0.7) + ($avg_rain * 0.3);
        } else {
            // No historical data or historical is 0, use BMKG directly
            $avg_rain = $bmkg_rainfall;
        }
    } elseif ($data_count == 0 || $avg_rain == 0) {
        // No data at all - ONLY use verified reports, NO assumptions
        if ($negative_report_count > 0) {
            // If there are verified kekeringan reports, use them to estimate
            $avg_rain = 15; // Low rainfall based on verified reports
        } elseif ($positive_report_count > 0) {
            // If there are verified positive reports, use them to estimate
            $avg_rain = 80; // Normal rainfall based on verified reports
        } else {
            // NO DATA AND NO REPORTS - Skip prediction for this dusun
            // Don't create prediction if there's no real data
            continue; // Skip to next dusun
        }
    }
    
    // Improved rain factor calculation
    // Normal rainfall in Indonesia: 50-200mm/month
    // < 20mm = very dry, 20-50mm = dry, 50-100mm = normal, > 100mm = wet
    // Rain factor: lower = better (less risk), higher = worse (more risk)
    if ($avg_rain < 20) {
        $rain_factor = 100; // Very high risk - extremely dry
    } elseif ($avg_rain < 50) {
        $rain_factor = 70; // High risk - dry
    } elseif ($avg_rain < 100) {
        $rain_factor = 25; // Low-medium risk - normal rainfall
    } elseif ($avg_rain < 200) {
        $rain_factor = 5; // Very low risk - good rainfall
    } else {
        $rain_factor = 0; // No risk - very wet
    }
    
    // Report factor calculation - synchronized with admin input and warga reports
    // Negative reports (kekeringan) increase risk significantly
    // Each verified kekeringan report = 15 points (more weight than before)
    $negative_factor = min(60, $negative_report_count * 15);
    
    // Positive reports (ketersediaan_air, kualitas_air, penampungan, sumur, pam) decrease risk significantly
    // These indicate water is available, so each report reduces risk by 8 points (max reduction: 40 points)
    $positive_reduction = min(40, $positive_report_count * 8);
    
    // Net report factor = negative - positive reduction
    $report_factor = max(0, $negative_factor - $positive_reduction);
    
    // Additional factors based on REAL DATA ONLY (no assumptions):
    // 1. If there are verified reports (either positive or negative), trust them more
    if ($total_verified_reports > 0) {
        // If we have verified reports, they are more reliable
        // Increase weight of report factor slightly
        $report_factor = $report_factor * 1.1; // 10% more weight
    }
    
    // 2. If many positive reports vs negative, strongly favor positive (REAL DATA)
    if ($positive_report_count > $negative_report_count && $positive_report_count >= 2) {
        // More positive reports = water is available, reduce risk more
        $report_factor = max(0, $report_factor - 15);
    }
    
    // REMOVED: No assumptions based on warga count without reports
    // Only use real verified data
    
    // Weight calculation - adjusted based on data availability
    // If we have manual data (admin input), trust it more
    // If we have verified reports, trust them more
    if ($manual_rainfall_today !== false && $manual_rainfall_today !== null) {
        // Admin has manually input data - give it very high weight
        // 75% rainfall (admin input), 25% reports
        $skor_risiko = ($rain_factor * 0.75) + ($report_factor * 0.25);
    } elseif ($total_verified_reports >= 3) {
        // Many verified reports - trust reports more
        // 60% rainfall, 40% reports
        $skor_risiko = ($rain_factor * 0.6) + ($report_factor * 0.4);
    } else {
        // Default: 70% rainfall, 30% reports
        $skor_risiko = ($rain_factor * 0.7) + ($report_factor * 0.3);
    }
    
    // Determine risk level with better thresholds
    // Adjusted thresholds to be more accurate
    if ($skor_risiko >= 60) {
        $tingkat_risiko = 'kering';
        $rekomendasi = 'Segera lakukan distribusi air tangki. Warga diharapkan menghemat penggunaan air.';
    } elseif ($skor_risiko >= 30) {
        $tingkat_risiko = 'waspada';
        $rekomendasi = 'Mulai persiapkan distribusi air. Monitor kondisi setiap hari.';
    } else {
        $tingkat_risiko = 'aman';
        $rekomendasi = 'Kondisi normal. Tetap jaga ketersediaan air.';
    }
    
    $faktor_utama = [];
    
    // Add desa information (data from admin input)
    $faktor_utama[] = 'Desa: ' . htmlspecialchars($dusun['nama_desa']) . 
                      (!empty($dusun['kode_desa']) ? ' (Kode: ' . htmlspecialchars($dusun['kode_desa']) . ')' : '');
    
    // Add rainfall information
    if ($avg_rain < 20) {
        $faktor_utama[] = 'Curah hujan sangat rendah (< 20mm)';
    } elseif ($avg_rain < 50) {
        $faktor_utama[] = 'Curah hujan rendah (20-50mm)';
    } elseif ($avg_rain < 100) {
        $faktor_utama[] = 'Curah hujan normal (' . round($avg_rain, 1) . 'mm)';
    } elseif ($avg_rain < 200) {
        $faktor_utama[] = 'Curah hujan tinggi (' . round($avg_rain, 1) . 'mm)';
    } else {
        $faktor_utama[] = 'Curah hujan sangat tinggi (' . round($avg_rain, 1) . 'mm)';
    }
    
    // Add data source information - ONLY REAL DATA
    if ($manual_rainfall_today !== false && $manual_rainfall_today !== null) {
        $faktor_utama[] = 'Data manual dari admin (' . round($manual_rainfall_today, 1) . 'mm) - Prioritas Tertinggi';
    } elseif ($bmkg_rainfall !== null) {
        $faktor_utama[] = 'Data real-time dari BMKG (' . round($bmkg_rainfall, 1) . 'mm)';
    } elseif ($data_count > 0 && $avg_rain > 0) {
        $faktor_utama[] = 'Data historis rata-rata (' . round($avg_rain, 1) . 'mm)';
    } elseif ($negative_report_count > 0 || $positive_report_count > 0) {
        $faktor_utama[] = 'Estimasi berdasarkan laporan terverifikasi';
    } else {
        // Should not reach here if we skip dusun without data
        $faktor_utama[] = 'Data tidak tersedia';
    }
    
    // Add report information (synchronized with admin verification)
    if ($negative_report_count > 0) {
        $faktor_utama[] = 'Ada ' . $negative_report_count . ' laporan kekeringan terverifikasi oleh admin';
    }
    
    if ($positive_report_count > 0) {
        $faktor_utama[] = 'Ada ' . $positive_report_count . ' laporan ketersediaan air terverifikasi oleh admin';
    }
    
    if ($total_verified_reports > 0) {
        $faktor_utama[] = 'Total ' . $total_verified_reports . ' laporan terverifikasi dalam 30 hari terakhir';
    }
    
    // Only show warga count if there are verified reports (real data connection)
    if ($total_warga > 0 && $total_verified_reports > 0) {
        $faktor_utama[] = 'Total ' . $total_warga . ' warga terdaftar, ' . $total_verified_reports . ' laporan terverifikasi';
    }
    
    // Don't add generic "Kondisi normal" - only show real data
    if (empty($faktor_utama)) {
        $faktor_utama[] = 'Data tidak lengkap';
    }
    
    // VALIDATION: Only create prediction if we have REAL DATA
    // Check if we have at least one of: manual data, BMKG data, historical data, or verified reports
    $has_real_data = false;
    if ($manual_rainfall_today !== false && $manual_rainfall_today !== null) {
        $has_real_data = true; // Admin input data
    } elseif ($bmkg_rainfall !== null && $bmkg_rainfall >= 0) {
        $has_real_data = true; // BMKG data
    } elseif ($data_count > 0 && $avg_rain > 0) {
        $has_real_data = true; // Historical data
    } elseif ($negative_report_count > 0 || $positive_report_count > 0) {
        $has_real_data = true; // Verified reports
    }
    
    // Skip dusun if no real data available - don't create fake predictions
    if (!$has_real_data) {
        continue; // Skip to next dusun
    }
    
    // Insert or update prediction - ONLY for dusun with real data
    $tanggal_prediksi = date('Y-m-d', strtotime('+7 days'));
    
    $stmt = $db->prepare("INSERT INTO prediksi_kekeringan 
                          (dusun_id, tanggal_prediksi, tingkat_risiko, skor_risiko, prediksi_hari, faktor_utama, rekomendasi) 
                          VALUES (?, ?, ?, ?, 7, ?, ?)
                          ON DUPLICATE KEY UPDATE 
                          tingkat_risiko = VALUES(tingkat_risiko),
                          skor_risiko = VALUES(skor_risiko),
                          faktor_utama = VALUES(faktor_utama),
                          rekomendasi = VALUES(rekomendasi)");
    
    if ($stmt->execute([
        $dusun['id'],
        $tanggal_prediksi,
        $tingkat_risiko,
        $skor_risiko,
        implode(', ', $faktor_utama),
        $rekomendasi
    ])) {
        $generated++;
    } else {
        $errors++;
    }
}

$success_msg = "Prediksi berhasil dibuat untuk $generated dusun";
if ($bmkg_updated > 0) {
    $success_msg .= " (Data BMKG diperbarui untuk $bmkg_updated dusun)";
}
if ($errors > 0) {
    $success_msg .= " ($errors error)";
}

$_SESSION['success'] = $success_msg;
redirect('index.php');

